/'****************************************************************************
The MIT License

Copyright (c) 2011, Ruchard D. Clark

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

http://www.opensource.org/licenses/mit-license.php
*****************************************************************************'/

#Ifndef _ASARRAY_BI_
#Define _ASARRAY_BI_

Namespace fbasarray

#Include "dataobj.bi"

Const MINTABLESIZE = 200 

'Node type used in linked list.
Type dnode
   dkey As dataobj       'The look up key.
   prevptr As dnode Ptr  'Pointer to previous element.
   nextptr As dnode Ptr  'Pointer to next element.
   ddata As dataobj      'Data.
End Type

'asarray object.   
Type asarray
   Private:
   dsize As Integer          'Current size of the table.
   dtable As dnode Ptr Ptr   'Hash table.
   initok As Integer         'Init flag.
   Declare Function hashhpjw(s As String) As Integer 'Hash function.
   Declare Sub Destroy () 'Clears memory.
   Public:
   Declare Constructor (tsize As Integer = MINTABLESIZE) 'Sets defaults for table. 
   Declare Destructor () 'Calls Destroy.
   Declare Property IsInit () As Integer 'Returns initok.
   Declare Property TableSize () As Integer 'Returns dsize.
   Declare Function FindKey(k As dataobj) As Integer 'Returns TRUE if key exists.
   Declare Function InsertKeyData(k As dataobj, dt As dataobj) As Integer 'Inserts data into table.
   Declare Function GetKeyData(k As dataobj, dt As dataobj) As Integer 'Returns data from table.
   Declare Function SetKeyData(k As dataobj, dt As dataobj) As Integer 'Sets data at key.
   Declare Function DeleteKeyData(k As dataobj) As Integer 'Deletes key and data.
End Type

'Hash function.
'From Book Compilers Principles, Techniques and Tools
'By Alfred V. Aho, Ravi Sethi and Jeffrey D. Ullman
Function asarray.hashhpjw(s As String) As Integer
   Dim As Integer h = 0, g, prime = 211

   If Len(s) > 0 Then
      For i As Integer = 1 To Len(s)
         h = (h Shl 4) + Asc(Mid(s, i, 1))
         If g = (h And &HF0) Then
            h = h ^ (g Shr 24)
            h = h ^ g
         EndIf
      Next
      h = h And prime
   End If
   
   Return h
End Function

'Deallocates the table.
Sub asarray.Destroy ()
   Dim As Integer cnt
   Dim As dnode Ptr dtsave, dtcurr
   
   'Make sure we have an active table.
   If dtable <> NULL Then
      If dsize > 0 Then
         'Iterate through the table and clear any strings.
         For i As Integer = 0 To dsize - 1
            'Make sure we have some data in the dnode.
            If dtable[i] <> NULL Then
               'Get the first node.
               dtcurr = dtable[i]
               'Walk down the nodes clearing any strings.
               Do While dtcurr <> NULL
                  'Clear key and data in node.
                  dtcurr->dkey = ""
                  dtcurr->ddata.ClearData
                  'Save the current node.
                  dtsave = dtcurr
                  'Get the next item in the bucket.
                  dtcurr = dtcurr->nextptr
               Loop 
               'Walk back down the nodes deallocating.
               dtcurr = dtsave->prevptr
               Do While dtcurr <> NULL
                  'Deallocate previous pointer.
                  DeAllocate dtcurr->nextptr
                  dtcurr = dtcurr->prevptr
               Loop 
               'Dealllocate last element.
               DeAllocate dtable[i]
            EndIf
         Next
         'Deallocate the table.
         DeAllocate dtable
      EndIf
   End If
End Sub

'Construct the table either size or min size.
Constructor asarray (tsize As Integer = MINTABLESIZE)
   
   initok = TRUE
   'Set the minimum size of the table.
   If tsize < MINTABLESIZE Then
      dsize = MINTABLESIZE
   Else
      dsize = tsize
   EndIf
   'Allocate table based on the size.
   dtable = Callocate(dsize, SizeOf(dnode Ptr))
   'Check to make sure all went well.
   If dtable = NULL Then
      initok = FALSE
      dsize = 0
   EndIf
End Constructor

'Destroys the object,
Destructor asarray ()
   Destroy
End Destructor

'Returns state of asarray object.
Property asarray.IsInit () As Integer
   Return initok
End Property

'Returns the current table size.
Property asarray.TableSize () As Integer
   Return dsize
End Property

'Returns True if key is found.
Function asarray.FindKey(k As dataobj) As Integer
   Dim As Integer ret = FALSE, hashadd
   Dim dtcurr As dnode Ptr
   
   'Make sure the table was created.
   If dtable <> NULL Then
      'Get the hash address.
      hashadd = hashhpjw(k)
      'Check to make sure the address is within the bounds of the table.
      If hashadd <= (dsize - 1) Then
         'Make sure have data in the bucket.
         If dtable[hashadd] <> NULL Then
            'Walk the bucket looking for string.
            dtcurr = dtable[hashadd]
            Do While dtcurr <> NULL
               'Do the names match?
               If dtcurr->dkey = k Then
                  'Found it.
                  ret = TRUE
                  Exit Do
               EndIf
               'Get the next item.
               dtcurr = dtcurr->nextptr
            Loop 
         EndIf
      EndIf
   End If
   
   Return ret
End Function

'Inserts a variable into the table. Returns TRUE if successful.
Function asarray.InsertKeyData(k As dataobj, dt As dataobj) As Integer
   Dim As Integer ret = TRUE, chk, idx
   Dim As UInteger hashadd 'Hash address.
   Dim tmpd As dnode Ptr Ptr 'Symbol table pointer.
   Dim As dnode Ptr newptr, currptr, savptr 'Data pointers.
    
   'Check to see if the symbol already exists.
   chk = FindKey(k)
   'Symbol already exists. 
   If chk = TRUE Then
      ret = FALSE
   Else
      'Get hash address.
      hashadd = hashhpjw(k)
      'If the address is larger than the table, expand the table.
      If hashadd > (dsize - 1) Then
         'The indexes run from 0 to n, so we need address + 1 elements.
         tmpd = ReAllocate(dtable, (hashadd + 1) * SizeOf(dnode Ptr))
         'Check for errors.
         If tmpd = NULL Then
            ret = FALSE
         Else
            'Set the new pointer.
            dtable = tmpd
            tmpd = NULL
            'Set the new size.
            dsize = hashadd + 1
         EndIf
      EndIf
      'Make sure everything is OK to this point.
      If ret = TRUE Then
         'Create a new data item.
         newptr = Callocate(1, SizeOf(dnode))
         'Make sure we could allocate memory.
         If newptr <> NULL Then
            newptr->ddata.SetData dt
            newptr->dkey = k
         Else
            ret = FALSE
         End If
         'Makse sure everything is ok.
         If ret = TRUE Then
            'Does the bucket already contain data?
            If dtable[hashadd] = NULL Then
               'Save the pointer into the symbol table.
               dtable[hashadd] = newptr
            Else
               'Walk down the list until we get to the end.
               currptr = dtable[hashadd]
               Do While currptr <> NULL
                  savptr = currptr
                  'Get the next node.
                  currptr = currptr->nextptr
               Loop 
               'Set the pointers.
               savptr->nextptr = newptr
               newptr->prevptr = savptr
            End If
         EndIf
      End If   
   EndIf
   
   Return ret   
End Function

'Returns TRUE if found and stores data in dt.
Function asarray.GetKeyData(k As dataobj, dt As dataobj) As Integer
   Dim As Integer ret = FALSE, hashadd
   Dim dtcurr As dnode Ptr
   
   'Make sure the table was created.
   If dtable <> NULL Then
      'Get the hash address.
      hashadd = hashhpjw(k)
      'Check to make sure the address is within the bounds of the table.
      If hashadd <= (dsize - 1) Then
         'Make sure have data in the bucket.
         If dtable[hashadd] <> NULL Then
            'Walk the bucket looking for string.
            dtcurr = dtable[hashadd]
            Do While dtcurr <> NULL
               'Do the names match?
               If dtcurr->dkey = k Then
                  dtcurr->ddata.GetData dt
                  ret = TRUE
                  Exit Do
               EndIf
               'Get the next item.
               dtcurr = dtcurr->nextptr
            Loop 
         EndIf
      EndIf
   End If
   
   Return ret
End Function

'Returns TRUE if found and stores data in table.
Function asarray.SetKeyData(k As dataobj, dt As dataobj) As Integer
   Dim As Integer ret = FALSE, hashadd
   Dim dtcurr As dnode Ptr
   
   'Make sure the table was created.
   If dtable <> NULL Then
      'Get the hash address.
      hashadd = hashhpjw(k)
      'Check to make sure the address is within the bounds of the table.
      If hashadd <= (dsize - 1) Then
         'Make sure have data in the bucket.
         If dtable[hashadd] <> NULL Then
            'Walk the bucket looking for string.
            dtcurr = dtable[hashadd]
            Do While dtcurr <> NULL
               'Do the names match?
               If dtcurr->dkey = k Then
                  dtcurr->ddata.SetData dt
                  ret = TRUE
                  Exit Do
               EndIf
               'Get the next item.
               dtcurr = dtcurr->nextptr
            Loop 
         EndIf
      EndIf
   End If
   
   Return ret
End Function

'Deletes key and data.
Function asarray.DeleteKeyData(k As dataobj) As Integer
   Dim As Integer ret = FALSE, hashadd
   Dim As dnode Ptr dtcurr, dtnext, dtprev
   
   'Make sure the table was created.
   If dtable <> NULL Then
      'Get the hash address.
      hashadd = hashhpjw(k)
      'Check to make sure the address is within the bounds of the table.
      If hashadd <= (dsize - 1) Then
         'Make sure have data in the bucket.
         If dtable[hashadd] <> NULL Then
            'Walk the bucket looking for string.
            dtcurr = dtable[hashadd]
            Do While dtcurr <> NULL
               'Do the names match?
               If dtcurr->dkey = k Then
                  'Get the next and previous pointers.
                  dtnext = dtcurr->nextptr
                  dtprev = dtcurr->prevptr
                  'Make sure we have a pointer.
                  If dtnext <> NULL Then
                     dtnext->prevptr = dtprev
                  EndIf
                  If dtprev <> NULL Then
                     dtprev->nextptr = dtnext
                  EndIf
                  'Clear data object.
                  dtcurr->ddata.ClearData
                  'Deallocate data.
                  DeAllocate dtcurr
                  ret = TRUE
                  Exit Do
               EndIf
               'Get the next item.
               dtcurr = dtcurr->nextptr
            Loop 
         EndIf
      EndIf
   End If
   
   Return ret
End Function


End Namespace
#EndIf