/'****************************************************************************
*
* Name: grfont.bi
*
* Synopsis: Graphic font object.
*
* Description: The graphic font object displays a set of graphics that correspond to
*              to the ascii character set. Each graphic must contain 256 characters
*              arranged in a 16x16 grid. Each character can be of any size, but all of 
*              the characters must be the same size. The first character corrsponds
*              to ascii character 0 while the last character corrsponds to ascii 255.
*              For example, the first row must corrspond to ascii characters 0 to 15.
*              The file must be in bmp format. See Aesomatica.bmp for an example of the 
*              format expected.
*
* Copyright 2010, Richard D. Clark
*
*                          The Wide Open License (WOL)
*
* Permission to use, copy, modify, distribute and sell this software and its
* documentation for any purpose is hereby granted without fee, provided that
* the above copyright notice and this license appear in all source copies. 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY OF
* ANY KIND. See http://www.dspguru.com/wol.htm for more information.
*
*****************************************************************************'/
Namespace grfont

#Include Once "fbgfx.bi"
   
Const NULL As Any Ptr = 0
Const FALSE = 0
Const TRUE = Not FALSE

Type coords
   x As Integer
   y As Integer
End Type

Type gfont
   Private:
   _img As fb.image Ptr
   _fname As String
   _imgw As Integer
   _imgh As Integer
   _chrw As Integer
   _chrh As Integer
   _grid(0 To 255) As coords
   Declare Sub _bmpload ()
   Declare Sub _Clear()
   Public:
   Declare Constructor()
   Declare Destructor ()
   Declare Property ImageWidth() As Integer
   Declare Property ImageHeight() As Integer
   Declare Property CharWidth() As Integer
   Declare Property CharHeight() As Integer
   Declare Function Load(fname As String) As Integer
   Declare Sub PrintChar(c As Integer, x As Integer, y As Integer)
   Declare Sub PrintChar(c As String, x As Integer, y As Integer)
   Declare Sub PrintString(s As String, x As Integer, y As Integer)
End Type
   
'From the FB help file.
Sub gfont._bmpload()
   Dim As Integer filenum, px = 0, py = 0

   '' open BMP file
   filenum = FreeFile()
   If Open( _fname For Binary Access Read As #filenum ) = 0 Then
      '' retrieve BMP dimensions
      Get #filenum, 19, _imgw
      Get #filenum, 23, _imgh
      Close #filenum
      'Calculate the character width.
      _chrw = _imgw / 16
      'Calculate the character height.
      _chrh = _imgh / 16
      'Fill in the grid coordinates.
      For y As Integer = 0 To 15
         For x As Integer = 0 To 15
            _grid(x + y * 16).x = px
            _grid(x + y * 16).y = py
            px += _chrw
         Next
         px = 0
         py += _chrh
      Next
      ' create image with BMP dimensions
      _img = ImageCreate( _imgw, Abs(_imgh) )
      If _img <> NULL Then
         '' load BMP file into image buffer
         If BLoad( _fname, _img ) <> 0 Then 
            ImageDestroy( _img )
         End If
      End If
   End If
End Sub

Sub gfont._Clear()
   If _img <> NULL Then
      ImageDestroy _img
   EndIf
   _img = NULL
   _fname = ""
   _imgw = 0
   _imgh = 0
   _chrw  = 0
   _chrh  = 0
   For i As Integer = 0 To 255
      _grid(i).x = 0
      _grid(i).y = 0
   Next
End Sub

Constructor gfont ()
   _Clear
End Constructor

Destructor gfont()
   _Clear
End Destructor

Property gfont.ImageWidth() As Integer
   Return _imgw
End Property

Property gfont.ImageHeight() As Integer
   Return _imgh
End Property

Property gfont.CharWidth() As Integer
   Return _chrw
End Property

Property gfont.CharHeight() As Integer
   Return _chrh
End Property

Function gfont.Load(fname As String) As Integer
   Dim As Integer ret
   
   If Len(Dir(fname)) = 0 Then
      Return FALSE
   Else
      _fname = fname
      _bmpload
      If _img = NULL Then
         Return FALSE
      Else
         Return TRUE
      EndIf
   EndIf
   
End Function

Sub gfont.PrintChar(c As Integer, x As Integer, y As Integer)
   Dim cimg As fb.image Ptr = ImageCreate(_chrw, _chrh)
   Dim As Integer x1, y1, x2, y2
   
   
   If _img <> NULL Then
      If ScreenPtr Then
         ScreenLock
         x1 = _grid(c).x
         x2 = _grid(c).x + _chrw - 1
         y1 = _grid(c).y
         y2 = _grid(c).y + _chrh - 1
         Get _img, (x1, y1) - (x2, y2), cimg
         'Get the image.
         Put (x, y), cimg, Trans
         ScreenUnLock     
      EndIf
   EndIf
   
   If cimg <> NULL Then
      ImageDestroy cimg
   EndIf
End Sub

Sub gfont.PrintChar(c As String, x As Integer, y As Integer)
   Dim As Integer cc
   
   cc = Asc(c)
   PrintChar cc, x, y
   
End Sub

Sub gfont.PrintString(s As String, x As Integer, y As Integer)
   Dim As Integer cc
   Dim As Integer xx = x, yy = y
   
   For i As Integer = 1 To Len(s)
      cc = Asc(s, i)
      PrintChar cc, xx, yy
      xx += _chrw
   Next   
End Sub
   
End Namespace
