/'****************************************************************************
*
* Name: robots.bas
*
* Synopsis: The classic robots game converted to FreeBasic.
*
* Description: This is the classic BSD Robots game converted to FreeBasic.
*              Use the number pad to move character to level exit. Robots will
*              move 1 space toward character using Manhattan distance. If 
*              robots collide, they will destroy each other leaving rubble. 
*              Rubble will destroy other robots. If a robot touches the 
*              character, the game is over. Each level adds more robots. The 
*              object is to get to successfully exit 100.
*
* Copyright 2009, Richard D. Clark
*
* Credits: Title graphic is from the icon collection:
*          http://icons.mysitemyway.com/gallery/post/3d-glossy-orange-orbs-icons-business/page-6/
*
*                          The Wide Open License (WOL)
*
* Permission to use, copy, modify, distribute and sell this software and its
* documentation for any purpose is hereby granted without fee, provided that
* the above copyright notice and this license appear in all source copies. 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY OF
* ANY KIND. See http://www.dspguru.com/wol.htm for more information.
*
*****************************************************************************'/
#Include "stringlist.bi"
#Include "vec.bi"
#Include "robot1.bi"
#Include "robot.bi"
#Include "robot3.bi"

'Object enum.
Enum eobject
	oempty = 0
	odebri
	oexit
	orobot
	opc
End Enum

'Robot directions
Enum rcompass
	rnorth
	reast
	rsouth
	rwest
End Enum

'Character directions
Enum compass
	north
	neast
	east
	seast
	south
	swest
	west
	nwest
End Enum

'Game conditions.
Enum gamecond
	condnone
	condlevel
	condlose
	condwin
End Enum

Const rversion = "0.1.3"
'Map size.
Const mapw = 80
Const maph = 58

'Key consts.
Const xk = Chr(255)
Const key_esc = Chr(27)
Const key_close = xk + "k"
const key_up = xk + "H"
const key_dn = xk + "P"
const key_rt = xk + "M"
const key_lt = xk + "K"

'Object icons.
Const pchar = "@"
Const debri = "*"
Const lexit = "<"
Const robot = "R"
'Object colors.
Const pcharclr = RGB(255, 126, 0)
Const debriclr = RGB(240, 220, 130)
Const exitclr = RGB(0, 191, 255)
Const robotclr = RGB(179, 27, 27)
Const white = RGB(255, 255, 255)
Const black = RGB(0, 0, 0)

'Direction vectors.
Dim Shared pmat(north To nwest) As vector.vec 
'Level array.
Dim Shared level(1 To mapw, 1 To maph) As eobject
'Player character.
Dim Shared pc As vector.vec
'Level.
Dim Shared As Integer lvl, score, numbots, useascii = FALSE
'High scores
Dim Shared hiscore As strlist.stringlist

Dim As Integer done = FALSE, gmdone = FALSE, move, ret
Dim As String skey, hscore, fscore = Space(10), flvl = Space(3) 
Dim pv As vector.vec
Dim As gamecond pgc, rgc

'Object type.
Type objtype
	Private:
	_objid As eobject
	_loc As vector.vec
	Public:
	Declare Constructor () 
	Declare Property OType (e As eobject)
	Declare Property OType () As eobject
	Declare Property ox (x As Integer)
	Declare Property ox () As Integer
	Declare Property oy (y As Integer)
	Declare Property oy () As Integer
	Declare Function MoveTowardChar (vc As vector.vec) As gamecond
End Type

'Set the default parms.
Constructor objtype ()
	_objid = oempty
	_loc.vx = 1
	_loc.vy = 1
End Constructor

'Sets and returns object type.
Property objtype.OType (e As eobject)
	_objid = e
End Property

Property objtype.OType () As eobject
	Return _objid
End Property

'Properties to set and return the x and y components.
Property objtype.ox (x As Integer)
	_loc.vx = x
End Property

Property objtype.ox () As Integer
	Return _loc.vx
End Property

Property objtype.oy (y As Integer)
	_loc.vy = y
End Property

Property objtype.oy () As Integer
	Return _loc.vy
End Property

'If robot, move toward character.
Function objtype.MoveTowardChar (vc As vector.vec) As gamecond
	Dim As vector.vec ovec = _loc'Location vectors
	Dim As vector.vec nvec
	Dim odist As Integer = 1000 'Distance variables.
	Dim ndist As Integer
	Dim ret As gamecond = condnone
	
	'Only robots can move.
	If _objid = orobot Then
		'Move toward the player.
		For i As Integer = north To nwest
			'Get the distance to pc.
			nvec = _loc + pmat(i)
			If (nvec.vx > 0) And (nvec.vx < mapw) Then
				If (nvec.vy > 0) And (nvec.vy < mapw - 2) Then
					ndist = nvec.Dist(vc)
					'Save values.
					If ndist < odist Then
						odist = ndist
						ovec = nvec 
					EndIf
				End If
			EndIf
		Next
		'Can't move onto level exit.
		If level(ovec.vx, ovec.vy) <> oexit Then
			'Robot ran into another robot.
			If level(ovec.vx, ovec.vy) = orobot Then
				'Set current robot to debri.
				_objid = odebri
				'Set map to debri.
				level(_loc.vx, _loc.vy) = odebri
				score += 1
				'Robot ran into debri.
			ElseIf level(ovec.vx, ovec.vy) = odebri Then
				level(_loc.vx, _loc.vy) = odebri
				_objid = odebri
				score += 1
			'Robot got the player.
			ElseIf level(ovec.vx, ovec.vy) = opc Then
				ret = condlose
			Else
				'Clear old location.
				level(_loc.vx, _loc.vy) = oempty
				'Move robot to new location.
				_loc = ovec
				level(_loc.vx, _loc.vy) = orobot
			EndIf
		EndIf
	End If
	
	Return ret
End Function

'Object array: Robots, debris, exit.
Dim Shared mapobj() As objtype

Type gl32type
    pixel As Integer
    clr As UInteger
End Type

Dim Shared As gl32type gchar(7, 7)  => { _
{(0,0),(0,0),(0,0),(1,16761035),(1,16761035),(0,0),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(1,16761035),(1,16761035),(0,0),(0,0),(0,0)}, _
{(0,0),(1,13158400),(1,13158400),(1,38400),(1,38400),(1,13158400),(1,13158400),(0,0)}, _
{(0,0),(1,13158400),(0,0),(1,38400),(1,38400),(0,0),(1,13158400),(0,0)}, _
{(0,0),(0,0),(1,38400),(1,38400),(1,38400),(1,38400),(0,0),(0,0)}, _
{(0,0),(0,0),(1,38400),(0,0),(0,0),(1,38400),(0,0),(0,0)}, _
{(0,0),(0,0),(1,38400),(0,0),(0,0),(1,38400),(0,0),(0,0)}, _
{(0,0),(0,0),(1,38400),(0,0),(0,0),(1,38400),(0,0),(0,0)}}

Dim Shared As gl32type gdebri(7, 7)  => { _
{(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(1,9535825),(0,0),(0,0),(0,0),(0,0)}, _
{(0,0),(1,9535825),(1,9535825),(1,9535825),(1,9535825),(1,9535825),(0,0),(0,0)}, _
{(0,0),(1,9535825),(1,8077056),(1,8077056),(1,9535825),(1,9535825),(1,9535825),(0,0)}, _
{(1,9535825),(1,8077056),(1,8077056),(1,9535825),(1,9535825),(1,9535825),(1,8077056),(0,0)}, _
{(1,9535825),(1,8077056),(1,9535825),(1,9535825),(1,8077056),(1,8077056),(1,9535825),(1,9535825)}}

Dim Shared As gl32type gexit(7, 7)  => { _
{(0,0),(1,32767),(1,32767),(1,32767),(1,32767),(1,32767),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,25800),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,25800),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,25800),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,16777215),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,25800),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,25800),(1,25800),(1,32767),(1,16776960),(1,32767),(0,0)}, _
{(0,0),(1,32767),(1,32767),(1,32767),(1,32767),(1,16776960),(1,32767),(0,0)}}

Dim Shared As gl32type grobot(7, 7)  => { _
{(0,0),(0,0),(1,16711680),(1,16711680),(1,16711680),(1,16711680),(0,0),(0,0)}, _
{(0,0),(0,0),(1,16711680),(1,16776960),(1,16776960),(1,16711680),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(1,16711680),(1,16711680),(0,0),(0,0),(0,0)}, _
{(0,0),(0,0),(0,0),(1,16711680),(1,16711680),(0,0),(0,0),(0,0)}, _
{(1,16776960),(1,16711680),(1,16711680),(1,16711680),(1,16711680),(1,16711680),(1,16711680),(1,16776960)}, _
{(0,0),(0,0),(1,16711680),(1,16711680),(1,16711680),(1,16711680),(0,0),(0,0)}, _
{(0,0),(0,0),(1,16711680),(0,0),(0,0),(1,16711680),(0,0),(0,0)}, _
{(0,0),(0,0),(1,16711680),(0,0),(0,0),(1,16711680),(0,0),(0,0)}}

'Returns a random number within range.
Function RandomRange(lowerbound As Integer, upperbound As Integer) As Integer
	Return Int((upperbound - lowerbound + 1) * Rnd + lowerbound)
End Function

'Set up the movement vectors.
Sub SetUpCharVectors
	pmat(north).vx = 0
	pmat(north).vy = -1
	pmat(neast).vx = 1
	pmat(neast).vy = -1
	pmat(east).vx = 1
	pmat(east).vy = 0
	pmat(seast).vx = 1
	pmat(seast).vy = 1
	pmat(south).vx = 0
	pmat(south).vy = 1
	pmat(swest).vx = -1
	pmat(swest).vy = 1
	pmat(west).vx = -1
	pmat(west).vy = 0
	pmat(nwest).vx = -1 
	pmat(nwest).vy = -1
End Sub

'Initialze the current level.
function InitLevel () As gamecond
	Dim As Integer rx, ry, rr
	Dim ret As gamecond = condnone
	
	lvl += 1
	'If player makes it past 100, then game is over.
	If lvl <= 100 Then
		numbots += 1
		'Clear current level array.
		For x As Integer = 1 To mapw
			For y As Integer = 1 To maph
				level(x, y) = oempty
			Next
		Next
		'Set pc position.
		 pc.vx = RandomRange (2, mapw - 2)
		 pc.vy = RandomRange (2, maph - 2)
		 level(pc.vx, pc.vy) = opc
		 Do
 			rx = RandomRange(2, mapw - 2)
		 	ry = RandomRange(2, maph - 2)
		 Loop Until level(rx, ry) = oempty
		 'Set the level exit.
		 level(rx, ry) = oexit
		 'Add some debris to the map.
		 For i As Integer = 1 To 10
 			rx = RandomRange(2, mapw - 2)
		 	ry = RandomRange(2, maph - 2)
		 	If level(rx, ry) = oempty Then
		 		level(rx, ry) = odebri
		 	EndIf
		 Next
		 'Set up the robots.
		 ReDim mapobj(1 To numbots)
		 For i As Integer = 1 To numbots
		 	Do 
		 		rx = RandomRange(2, mapw - 2)
		 		ry = RandomRange(2, maph - 2)
		 	Loop Until level(rx, ry) = oempty
		 	mapobj(i).ox = rx
		 	mapobj(i).oy = ry
		 	mapobj(i).OType = orobot
		 	level(rx, ry) = orobot
		 Next
	Else
		ret = condwin
		score += 100
	End If
	
	Return ret
End Function

'Draws the glyph.
Sub PrintGlyph(obj() As gl32type, grow As Integer, gcol As Integer)
    Dim As Integer i, j, row, col, scrw
    Dim framebuffer As UInteger Ptr

    row = (grow - 1) * (Ubound(obj, 1) + 1)
    col = (gcol - 1) * (Ubound(obj, 2) + 1)


    framebuffer = Screenptr
    If framebuffer Then
        Screeninfo scrw
        For i = Lbound(obj, 1) To Ubound(obj, 1)
            For j = Lbound(obj, 2) To Ubound(obj, 2)
                If obj(i, j).pixel = 1 Then
                    Poke UInteger, framebuffer + ((row + i) * scrw) + (col + j), obj(i, j).clr
                End If
            Next
        Next
    End If
End Sub

'Draws level to screen.
Sub DrawLevel ()
	
	ScreenLock
	Cls
	If useascii = TRUE Then
		'Draw the map objects.
		For x As Integer = 1 To mapw
			For y As Integer = 1 To maph
				If level(x, y) = oexit Then
					Locate y, x
					Color exitclr
					Print lexit;
				EndIf
				If level(x, y) = odebri Then
					Locate y, x
					Color debriclr
					Print debri;
				EndIf
				If level(x, y) = orobot Then
					Locate y, x
					Color robotclr
					Print robot;
				EndIf
				If level(x, y) = opc Then
					Locate y, x
					Color pcharclr
					Print pchar;
				EndIf
			Next
		Next
	Else
		'Draw the map objects.
		For x As Integer = 1 To mapw
			For y As Integer = 1 To maph
				If level(x, y) = oexit Then
					PrintGlyph gexit(), y, x
				EndIf
				If level(x, y) = odebri Then
					PrintGlyph gdebri(), y, x
				EndIf
				If level(x, y) = orobot Then
					PrintGlyph grobot(), y, x
				EndIf
				If level(x, y) = opc Then
					PrintGlyph gchar(), y, x
				EndIf
			Next
		Next
	End If
	Locate maph + 1, 1
	Color white
	Print "ROBOTS  Level: " & lvl & " Score: " & score;
	ScreenUnLock
End Sub

Sub DoTitle ()
	Dim clr As UInteger
	Dim As String tt = "Clark Productions", ss
	
	For x As Integer = 0 To robot1w - 1
		For y As Integer = 0 To robot1h - 1
			clr = robot1(x + y * robot1w)
			Locate y + 1, x + 1
			Color clr
			Print Chr(219);
		Next
	Next
	Color black, white
	Locate maph, (mapw / 2) - (Len(tt) / 2)
	Print tt;
	Sleep	
	Color white, black
	Cls
	tt = "Robots"
	Locate 2, (mapw / 2) - (Len(tt) / 2)
	Print tt
	Print:Print
	Print " Robots is based on the classic BSD Robot game. The goal is to move the "
	Print
	Print " character to the level exit. After each move, the robots will move one space"
	Print
	Print " toward the character. If a robot touches the character, the game is over. If "
	Print
	Print " a robot collides with another robot, the colliding robot will turn to debri."
	Print
	Print " Debri will also destroy a robots, creating more debri. For each robot destroyed"
	Print
	Print " you will score 1 point. You will score 10 points when you reach the level exit."
	Print 
	Print
	Print " Move using the numpad with numlock on. You can move in all 8 cardinal "
	Print
	Print " directions. Press 't' to teleport to a random position. Robots can move"
	Print  
	Print " any direction toward the character as well. Each level has one more robot"
	Print
	Print " than the previous level. If you make it to level 100 you win the game, "
	Print
	Print " scoring an additonal 100 points. You can exit using the ESC key."
	Print
	Print
	Print " You can play Robots in either ascii mode or graphical mode:"
	Print
	Print
	Print " Ascii: Robot: ";
	Color  robotclr
	Print robot;
	Color white 
	Print " Debri: ";
	Color debriclr 
	Print debri;
	Color white 
	Print " Exit: ";
	Color exitclr 
	Print lexit;
	Color white 
	Print " Character: ";
	Color pcharclr 
	Print pchar
	Print
	Print
	Color white
	Print " Graphical: Robot:"
	PrintGlyph grobot(), 37, 20
	Locate 37, 22
	Print " Debri:"
	PrintGlyph gdebri(), 37, 30
	Locate 37, 32
	Print " Exit:"
	PrintGlyph gexit(), 37, 39
	Locate 37, 41
	Print " Character:"
	PrintGlyph gchar(), 37, 53
	Print
	Print
	Print " Play ascii [A] or graphical [G]?"
	Do
		ss = InKey
		ss = UCase(ss)
		Sleep 1
	Loop Until (ss = "A") Or (ss = "G")
	If ss = "A" Then useascii = TRUE
	Do: Loop Until InKey = ""
End Sub

'Move player character.
function MoveCharacter (v As vector.vec) As gamecond
	Dim ret As gamecond = condnone
	
	'Make sure that the move is in bounds.
	If (v.vx > 0) And (v.vx <= mapw) Then
		If (v.vy > 0) And (v.vy <= maph - 2) Then
			'Cant move through debri.
			If level(v.vx, v.vy) <> odebri Then
				'Check the map.
				If level(v.vx, v.vy) = orobot Then
					ret = condlose
				ElseIf level(v.vx, v.vy) = oexit Then
					ret = condlevel
				Else
					level(pc.vx, pc.vy) = oempty
					pc = v
					level(pc.vx, pc.vy) = opc
				EndIf
			End If
		EndIf
	EndIf
	Return ret	
End Function

'Moves robots.
Function MoveAllRobots () As gamecond
	Dim ret As gamecond
	
	For i As Integer = 1 To numbots
		If mapobj(i).OType = orobot Then
			ret = mapobj(i).MoveTowardChar(pc)
			If ret = condlose Then
				Exit For
			EndIf
		EndIf
	Next
	
	Return ret
End Function

Sub DrawLoseScreen
	Dim clr As UInteger
	
	For x As Integer = 0 To robot2w - 1
		For y As Integer = 0 To robot2h - 1
			clr = robot2(x + y * robot2w)
			Locate y + 1, x + 1
			Color clr
			Print Chr(219);
		Next
	Next
	Sleep
End Sub

Sub DrawWinScreen
	Dim clr As UInteger
	
	For x As Integer = 0 To robot3w - 1
		For y As Integer = 0 To robot3h - 1
			clr = robot3(x + y * robot3w)
			Locate y + 1, x + 1
			Color clr
			Print Chr(219);
		Next
	Next
	Sleep
End Sub

Sub DoTeleport
	Dim As Integer rx, ry
	
	Do
		rx = RandomRange(1, mapw)
		ry = RandomRange(1, maph)
	Loop Until level(rx, ry) = oempty
	level(pc.vx, pc.vy) = oempty
	pc.vx = rx
	pc.vy = ry
	level(pc.vx, pc.vy) = opc
End Sub


Randomize Timer
Screen 18, 32
WindowTitle "Robots v" & rversion
Width mapw, maph + 2
DoTitle
SetUpCharVectors
Do
	lvl = 0
	score = 0
	numbots = 9	
	InitLevel
	DrawLevel
	done = FALSE
	Do
		skey = InKey
		If Len(skey) > 0 Then
			pgc = condnone
			rgc = condnone
			move = FALSE
			'Clicked on close button.
			If skey = key_close Then
				done = TRUE
			EndIf
			'Hit escape.
			If skey = key_esc Then
				done = TRUE
			EndIf
			'Moved character.
			If (skey = key_up) Or (skey = "8") Then
				pv = pc + pmat(north)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If (skey = key_dn) Or (skey = "2") Then
				pv = pc + pmat(south)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If (skey = key_rt) Or (skey = "6") Then
				pv = pc + pmat(east)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If (skey = key_lt) Or (skey = "4") Then
				pv = pc + pmat(west)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If skey = "7" Then
				pv = pc + pmat(nwest)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If skey = "9" Then
				pv = pc + pmat(neast)
				pgc = MoveCharacter(pv)
				move = TRUE
			EndIf
			If skey = "1" Then
				pv = pc + pmat(swest)
				pgc = MoveCharacter(pv)
				move = TRUE
			End If
			If skey = "3" Then
				pv = pc + pmat(seast)
				pgc = MoveCharacter(pv)
				move = TRUE
			End If
			If (skey = "t") Or (skey = "T") Then
				DoTeleport
				move = TRUE
			EndIf
			If move = TRUE Then
				If pgc = condlevel Then
					score += 10
					pgc = InitLevel ()
					If pgc = condwin Then
						done = TRUE
					EndIf
					DrawLevel
				ElseIf pgc = condlose Then
					done = TRUE
				Else
					rgc = MoveAllRobots()
					If rgc = condlose Then
						pgc = condlose
						done = TRUE
					Else
						DrawLevel
					EndIf
				EndIf
			EndIf
		EndIf
		Sleep 1
	Loop Until done = TRUE
	
	'Display the lose or win screen.
	If pgc = condlose Then
		DrawLoseScreen
	ElseIf pgc = condwin Then
		DrawWinScreen
	EndIf
	Cls
	
	If (pgc = condlose) Or (pgc = condwin) Then
		ret = hiscore.LoadFromFile("rhiscore.dat")
		If score > 0 Then
			'Show the high scores.
			RSet fscore, Str(score)
			RSet flvl, Str(lvl)
			hscore = "Score: " & fscore & " Level: " & flvl
			If ret = FALSE Then
				ret = hiscore.Add(hscore)
			Else
				ret = hiscore.Add(hscore)
				hiscore.SortAscending = FALSE
				hiscore.Sort
				If hiscore.Count > maph - 1 Then
					ret = hiscore.DeleteItem(hiscore.Count - 1)
				EndIf
			EndIf
			ret = hiscore.SaveToFile("rhiscore.dat")
		End If
		Color white, black
		Locate 1, (mapw / 2) - (Len("High Scores") / 2)
		Print "High Scores"
		For i As Integer = 0 To hiscore.Count - 1
			hscore = hiscore.Strings(i)
			Print hscore
		Next
		
		Sleep
	End If
	
	Cls
	Print
	Print "Do you want to play again? [Y/N]"
	Do
		skey = InKey
		skey = UCase(skey)
	Loop Until (skey = "Y") Or skey = "N"
	If skey = "N" Then
		gmdone = TRUE
	EndIf
Loop Until gmdone = TRUE
